/*
 * snes9express
 * skins.cc
 * Copyright (C) 2004  David Nordlund
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * For further details, please read the included COPYING file,
 * or go to http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <iostream>
#include <sstream>
#include <set>
#include <vector>
#include <string>
#include "frend.h"
#include "interface.h"
#include "skins.h"

s9x_SkinSelector::s9x_SkinSelector(fr_Element*parent):
#ifdef S9X_SKINABLE
fr_PulldownMenu(parent, "Select a skin")
#else
fr_PulldownMenu(parent, "Skin support not compiled in")
#endif
{
#ifdef S9X_SKINABLE
  char *ds = getenv("SNES9EXPRESS_DEFAULT_SKIN");
  if(ds)
    defaultskin = ds;
  LoadPaths();
  FillMenu();
#else
  SetEditable(false);
#endif
}

s9x_SkinSelector::~s9x_SkinSelector()
{
}


void s9x_SkinSelector::setDefaultSkinFromFile(const std::string& f)
{
  std::ifstream ds;
  ds.open(f.c_str(), std::ifstream::in);
  if(ds.is_open())
    ds >> defaultskin;
  ds.close();
}

// Get the list of available skins
void s9x_SkinSelector::LoadPaths() {
#ifdef S9X_SKINABLE
  char *s9xskindir = getenv("SNES9EXPRESS_SKINS");
  const char* suffix=".s9xskin";
  std::string Home(fr_HomeDir());
  std::vector<std::string> DirectoryNames;
  struct dirent *DirEntry;
  DIR *D;

  DirectoryNames.push_back(".");
  DirectoryNames.push_back(Home + "/" S9X_SUBDIR);
  if((s9xskindir)&&(fr_DirExists(s9xskindir)))
    DirectoryNames.push_back(s9xskindir);
#ifdef S9XDATADIR
  DirectoryNames.push_back(S9XDATADIR);
#endif


  skinPaths.clear();
  std::vector<std::string>::iterator i=DirectoryNames.begin(), e=DirectoryNames.end();
  PathMap::iterator nopath = skinPaths.end();
  for(;i!=e;i++)
  {
      std::string Path(*i);
      D = opendir(Path.c_str());
      if(!D)
	  continue;
      for(DirEntry = readdir(D); DirEntry; DirEntry = readdir(D))
      {
	if(DirEntry->d_name[0]=='.') continue;
	char *dot = strrchr(DirEntry->d_name, '.');
	if((!defaultskin.size())&&(!strcmp("defaultskin", DirEntry->d_name)))
	  setDefaultSkinFromFile(Path + "/" + DirEntry->d_name);
	if(!dot) continue;
	if(strcmp(dot, suffix)!=0) continue;
	std::string skinname(DirEntry->d_name, dot - DirEntry->d_name);
	if(skinPaths.find(skinname)==nopath)
          skinPaths[skinname] = Path + "/" + skinname + suffix;
      }
      closedir(D);
  }
#endif
}

void s9x_SkinSelector::FillMenu()
{
#ifdef S9X_SKINABLE
  std::set<std::string> sortedSkins;
  fr_MenuItem *mi;
  int idx = 2;

  PathMap::const_iterator i = skinPaths.begin(), e = skinPaths.end();
  for(;i!=e;i++)
        sortedSkins.insert((*i).first);

  mi = new fr_MenuItem(this, "Default");
  mi->SetTooltip("Current default: " + (defaultskin.size()?defaultskin:"Skinless"));
  mi = new fr_MenuItem(this, "Skinless");
  mi->Args << "--skin=-" << "--skin=";
  mi->SetTooltip("The no frills interface");
  std::set<std::string>::const_iterator si=sortedSkins.begin(), se=sortedSkins.end();
  for(;si!=se; si++,idx++)
  {
    std::string skinname(*si);
    mi = new fr_MenuItem(this, skinname.c_str());
    mi->Args << ("--skin="+skinname) << ("--gui="+skinname);
    mi->SetTooltip("Use the " + skinname + " skin");
  }
#endif
}

std::string s9x_SkinSelector::GetSelectedSkinName()
{
#ifdef S9X_SKINABLE
  int i = GetIndex();
  if(i==0)
    return defaultskin;
  else if(i < 2)
    return "";
  fr_ToggleInGroup *s = GetSelected();
  if(s)
    return s->GetName();
#endif
  return "";
}

std::string s9x_SkinSelector::GetSelectedSkinPath()
{
#ifdef S9X_SKINABLE
  std::string s = GetSelectedSkinName();
  if(s.size()<1) return s;
  PathMap::iterator e = skinPaths.end(), f = skinPaths.find(s);
  if(e!=f)
     return (*f).second;
#endif
  return "";
}



s9x_SkinSection::s9x_SkinSection(s9x_Skin*parent):
skin(parent)
{
}

s9x_SkinSection::~s9x_SkinSection()
{
}

bool s9x_SkinSection::has(const std::string& name) const
{
  std::map<std::string,std::string>::const_iterator i = properties.find(name);
  return i != properties.end();
}

int s9x_SkinSection::getInt(const std::string& name) const
{
  std::map<std::string,std::string>::const_iterator i = properties.find(name);
  if(i==properties.end())
    return 0;
  std::string s = (*i).second;
  return atoi(s.c_str());
}

fr_Colors s9x_SkinSection::getColors(const std::string& name) const
{
  std::map<std::string,std::string>::const_iterator i = properties.find(name);
  fr_Colors clr;
  if(i==properties.end())
    return clr;
  std::istringstream ss(i->second);
  std::string s;
  int rgb, r, g, b;
  while(ss >> s)
  {
    const char*c = s.c_str();
    if(c[0]=='#')
    {
      switch(s.size())
      {
      case 4: // #RGB
         rgb = strtol(&(c[1]), 0, 0x10);
         r = rgb / 0x100 * 0x11;
         g = (rgb & 0x0f0) / 0x10 * 0x11;
         b = (rgb & 0xf) * 0x11;
         clr += (r * 0x10000 + g * 0x100 + b);
         break;
      case 7: // #RRGGBB
         rgb = strtol(&(c[1]), 0, 0x10);
         clr += rgb;
         break;
      }
    }
  }
  return clr;
}

fr_Image* s9x_SkinSection::getImage(const std::string& imgname) const
{
  std::map<std::string,std::string>::const_iterator i = properties.find(imgname);
  if(i==properties.end())
    return NULL;
  return skin->getImage(i->second);
}

s9x_Skin::s9x_Skin(s9x_Interface*parent, const std::string& SkinFile):
Parent(parent),
skinfilename(SkinFile)
{
#ifdef S9X_SKINABLE
  skinr = new s9x_SkinReader(SkinFile);
  load();
#endif
}

s9x_Skin::~s9x_Skin()
{
#ifdef S9X_SKINABLE
  std::map<std::string,s9x_SkinSection*>::iterator i = sections.begin(), e = sections.end();
  while(i!=e)
    {
      delete (*i).second;
      i++;
    }
  std::map<std::string,fr_Image*>::iterator i2 = images.begin(), e2 = images.end();

  while(i2!=e2)
    {
      delete (*i2).second;
      i2++;
    }
  delete skinr;
#endif
}

std::string s9x_Skin::getName() const
{
#ifdef S9X_SKINABLE
  std::string s = skinfilename;
  unsigned int p = s.rfind('/');
  if(p!=s.npos)
	s.erase(0, p+1);
  return s;
#else
  return "";
#endif
}

void s9x_Skin::load()
{
#ifdef S9X_SKINABLE
  std::istream& SkinStream = skinr->getIstream(PROG ".skin");
  try
  {
	std::string sec;

	while(SkinStream >> sec)
	{
		s9x_SkinSection* ss = new s9x_SkinSection(this);
		try
		{
			SkinStream >> ss;
		}
		catch(char*e)
		{
			delete ss;
			throw e;
		}
		sections[sec] = ss;
	}
  } catch(char*e) {
	delete &SkinStream;
	throw e;
  }
  delete &SkinStream;
#endif
}

fr_Image* s9x_Skin::getImage(const std::string& imgname)
{
#ifdef S9X_SKINABLE
  std::map<std::string,fr_Image*>::const_iterator i = images.find(imgname);
  if(i!=images.end())
    return i->second;
  int filesize = skinr->getFileSize(imgname);
  if(filesize)
  {
    fr_Image* newImage = new fr_Image(skinr->seekFile(imgname), filesize);
    images[imgname] = newImage;
    return newImage;
  }
#endif
  return (fr_Image*)0;
}

// Read an element from the stream
std::istream& operator>> (std::istream& is, s9x_SkinSection* ss) {
#ifdef S9X_SKINABLE
   int s;
   std::string data, key, val;
   is >> data;
   if (data != "{")
     throw "skin format error, expected '{'";
   while(is >> data)
   {
      if((data=="}")||(data=="};"))
	return is;
      s = data.size() - 1;
      if(data[s] != ':')
	throw "skin format error, expected ':'";
      key = data.substr(0, s);
      val = "";
      while(is >> data)
      {
        s = data.size() - 1;
        if(data[s] != ';')
          val += data + " ";
        else
        {
          val += data.substr(0, s);
          break;
        }
      }
      ss->set(key, val);
  }
#endif
  return is;
}
