require 'abstract_unit'

class OptionMergerTest < Test::Unit::TestCase
  def setup
    @options = {:hello => 'world'}
  end

  def test_method_with_options_merges_options_when_options_are_present
    local_options = {:cool => true}

    with_options(@options) do |o|
      assert_equal local_options, method_with_options(local_options)
      assert_equal @options.merge(local_options),
        o.method_with_options(local_options)
    end
  end

  def test_method_with_options_appends_options_when_options_are_missing
    with_options(@options) do |o|
      assert_equal Hash.new, method_with_options
      assert_equal @options, o.method_with_options
    end
  end

  def test_method_with_options_allows_to_overwrite_options
    local_options = {:hello => 'moon'}
    assert_equal @options.keys, local_options.keys

    with_options(@options) do |o|
      assert_equal local_options, method_with_options(local_options)
      assert_equal @options.merge(local_options),
        o.method_with_options(local_options)
      assert_equal local_options, o.method_with_options(local_options)
    end
    with_options(local_options) do |o|
      assert_equal local_options.merge(@options),
        o.method_with_options(@options)
    end
  end

  def test_nested_method_with_options_containing_hashes_merge
    with_options :conditions => { :method => :get } do |outer|
      outer.with_options :conditions => { :domain => "www" } do |inner|
        expected = { :conditions => { :method => :get, :domain => "www" } }
        assert_equal expected, inner.method_with_options
      end
    end
  end

  def test_nested_method_with_options_containing_hashes_overwrite
    with_options :conditions => { :method => :get, :domain => "www" } do |outer|
      outer.with_options :conditions => { :method => :post } do |inner|
        expected = { :conditions => { :method => :post, :domain => "www" } }
        assert_equal expected, inner.method_with_options
      end
    end
  end

  def test_nested_method_with_options_containing_hashes_going_deep
    with_options :html => { :class => "foo", :style => { :margin => 0, :display => "block" } } do |outer|
      outer.with_options :html => { :title => "bar", :style => { :margin => "1em", :color => "#fff" } } do |inner|
        expected = { :html => { :class => "foo", :title => "bar", :style => { :margin => "1em", :display => "block", :color => "#fff" } } }
        assert_equal expected, inner.method_with_options
      end
    end
  end
  
  def test_nested_method_with_options_using_lamdba
    local_lamdba = lambda { { :lambda => true } }
    with_options(@options) do |o|
      assert_equal @options.merge(local_lamdba.call),
        o.method_with_options(local_lamdba).call
    end
  end

  # Needed when counting objects with the ObjectSpace
  def test_option_merger_class_method
    assert_equal ActiveSupport::OptionMerger, ActiveSupport::OptionMerger.new('', '').class
  end

  private
    def method_with_options(options = {})
      options
    end
end
