
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
# include <winsock2.h>
# define MSG_DONTWAIT 0
#else
# include <sys/types.h>
# include <sys/time.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <sys/fcntl.h>
# include <netdb.h>
# include <unistd.h>
# define closesocket close
#endif

#include <errno.h>
#include <glib.h>

#include "myx_ser_aux_functions.h"
#include "input.h"

int client_connect(const char *host, int port)
{
  struct sockaddr_in addr;
  struct hostent *hptr;
  struct in_addr **aptr;
  int sock;

  /* get a socket */
  sock= socket(PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
  {
    g_error("could not create socket: %s", g_strerror(errno));
    return -1;
  }

  /* resolve name */
  memset(&addr, sizeof(struct sockaddr_in), 0);
  addr.sin_family = AF_INET;
  addr.sin_port = htons(port);

  hptr = gethostbyname(host);
  if (!hptr)
  {
    closesocket(sock);
    return -1;
  }

  /* try all addresses */
  for (aptr = (struct in_addr**)hptr->h_addr_list; *aptr != NULL; aptr++) {
    memcpy(&addr.sin_addr, *aptr, sizeof(**aptr));
    if (connect(sock, (struct sockaddr*)&addr, sizeof(addr)) == 0) {
      break;
    }
  }

  if (!*aptr)
  {
    g_error("could not connect socket to %s:%i: %s", 
            host, port, strerror(errno));
    closesocket(sock);
    return -1;
  }
  
  return sock;
}


int server_bind(int port)
{
  int sock;
  struct sockaddr_in addr;

  sock= socket(PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
  {
    g_error("could not create socket: %s", g_strerror(errno));
    return -1;
  }

  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;
  addr.sin_port = htons(port);
  if (bind(sock, (struct sockaddr*)&addr, sizeof(addr)) < 0)
  {
    g_error("could not bind socket to port %i: %s", port, g_strerror(errno));
    return -1;
  }

  return sock;
}



TEST_RECORDS *make_example_data()
{
  TEST_RECORDS *recs= g_malloc(sizeof(TEST_RECORDS));
  int i,j;
  recs->record_group_num= 5;
  recs->record_group= g_malloc(sizeof(TEST_RECORD_GROUP)*recs->record_group_num);
  for (i=0; i < recs->record_group_num; i++)
  {
    recs->record_group[i].name= g_strdup_printf("GROUP %i", i);
    recs->record_group[i].records_num= rand()%10;
    recs->record_group[i].records= g_malloc(sizeof(TEST_RECORD)*recs->record_group[i].records_num);
    for (j=0;j<recs->record_group[i].records_num;j++)
    {
      TEST_RECORD *rec= recs->record_group[i].records+j;
      rec->name= g_strdup_printf("Record %i", j);
      rec->value= 12345;
      rec->num= ENUM_two;
      rec->description= g_strdup_printf("Record %i of group %i", j,i);
      rec->group= recs->record_group+i;
    }
  }
  return recs;
}


void send_stuff(int sock)
{
  MYX_SFD *sfd= myx_sfd_from_fd(sock);
  
  if (ser_data_records(sfd, make_example_data()) < 0)
    g_error("ERROR SENDING DATA");
  else
    g_message("sent data");

  myx_free_sfd(sfd);
}


int compare(TEST_RECORDS *a, TEST_RECORDS *b)
{
  int i,j;
  if (a->record_group_num != b->record_group_num)
    return 0;
  for (i=0; i < a->record_group_num; i++)
  {
    if (strcmp(a->record_group[i].name,b->record_group[i].name)!=0)
      return 0;
    if (a->record_group[i].records_num!=b->record_group[i].records_num)
      return 0;
    for (j=0;j<a->record_group[i].records_num;j++)
    {
      TEST_RECORD *arec= a->record_group[i].records+j;
      TEST_RECORD *brec= b->record_group[i].records+j;
      if (strcmp(arec->name, brec->name)!=0)
        return 0;
      if (arec->value!=brec->value)
        return 0;
      if (arec->num!= brec->num)
        return 0;
      if (strcmp(arec->description,brec->description)!=0)
        return 0;
      if (arec->group!= a->record_group+i)
        return 0;
      if (brec->group!= b->record_group+i)
        return 0;
    }
  }
  return 1;
}


void receive_stuff(int sock)
{
  MYX_SFD *sfd= myx_sfd_from_fd(sock);
  TEST_RECORDS *recs= g_malloc0(sizeof(TEST_RECORDS));
  char *id= NULL;
  
  // read in the record identifier
  unser_string(sfd, &id);
  if (!id)
  {
    g_error("error receiving data");
    return;
  }
  g_message("receiving '%s'...", id);
  if (unser_data_records(sfd, recs) < 0)
    g_error("ERROR RECEIVING DATA");
  else
    g_message("got data");

  if (!compare(make_example_data(), recs))
    g_message("RECEIVED DATA DOES NOT MATCH WHAT WAS EXPECTED");
  else
    g_message("and it matched");
  
  myx_free_sfd(sfd);
}



int main(int argc, char **argv)
{
  int server= argc<2?0:atoi(argv[1]);
  int sock;
  
  if (server)
  {
    sock= server_bind(5555);
    if (sock < 0)
      exit(1);
    if (listen(sock, 5) < 0)
    {
      g_error("could not listen to socket: %s", g_strerror(errno));
      exit(1);
    }
    for (;;)
    {
      int s, l;
      struct sockaddr_in peer;
      l= sizeof(peer);
      g_message("waiting connection...");
      s= accept(sock, (struct sockaddr*)&peer, &l);
      if (s < 0)
      {
        perror(errno);
        exit(1);
      }
      g_message("got connection");
      send_stuff(s);
      closesocket(s);
      break;
    }
  }
  else
  {
    sock= client_connect("127.0.0.1", 5555);
    if (sock < 0)
      exit(1);
    receive_stuff(sock);
  }
  return 0;
}

