# Copyright (C) 2006 Canonical Limited.
# Authors: David Allouche <david@allouche.net>
#          Aaron Bentley <aaron.bentley@utoronto.ca>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
from bzrlib.errors import BzrCommandError
from bzrlib.commands import Command, register_command
from bzrlib.branch import Branch, BranchFormat
from bzrlib.bzrdir import BzrDir, BzrDirFormat
from bzrlib.transport import get_transport
from bzrlib.builtins import cmd_update
from bzrlib.trace import note
from bzrlib.workingtree import WorkingTree

class cmd_switch(Command):
    """Set the branch of a lightweight checkout and update.  <BZRTOOLS>"""

    takes_args = ['to_location']

    def run(self, to_location):
        to_branch = Branch.open(to_location)
        tree_location = '.'
        self._switch(tree_location, to_branch)
        # need to re-open tree to get the new branch
        tree = WorkingTree.open_containing(tree_location)[0]
        self._update(tree)

    def _switch(self, tree_location, to_branch):
        tree = WorkingTree.open_containing(tree_location)[0]
        tree.lock_write()
        try:
            self._check_switch_branch_format(tree_location)
            self.set_branch_location(tree.bzrdir, to_branch.base)
            note('Switched to branch: %s' % (to_branch.base,))
        finally:
            tree.unlock()

    def _update(self, tree):
        tree.lock_write()
        try:
            if tree.last_revision() == tree.branch.last_revision():
                assert tree.branch.get_master_branch() is None, (
                    "switch tried to update a fat checkout")
                note("Tree is up to date.")
                return
            tree.update()
            note('Updated to revision %d' %
                 (tree.branch.revision_id_to_revno(tree.last_revision()),))
        finally:
            tree.unlock()

    def _check_switch_branch_format(self, url):
        transport = get_transport(url)
        format = BzrDirFormat.find_format(transport)
        format_string = format.get_format_string()
        if not format_string.startswith("Bazaar-NG meta directory, "):
            raise BzrCommandError(
                'The switch command can only be used on a light checkout.\n'
                'Expected metadir, found %s at %s' % (
                format_string.strip(), transport.base))
        control = BzrDir.open_containing_from_transport(transport)[0]
        branch_format = BranchFormat.find_format(control)
        format_string = branch_format.get_format_string()
        if not format_string.startswith("Bazaar-NG Branch Reference Format "):
            raise BzrCommandError(
                'The switch command can only be used on a light checkout.\n'
                'Expected branch reference, found %s at %s' % (
                format_string.strip(), transport.base))
        if not format_string == "Bazaar-NG Branch Reference Format 1\n":
            raise BzrCommandError(
                'Unsupported: %r' % (format_string.strip(),))        

    def set_branch_location(self, control, location):
        """Set location value of a branch reference.

        The branch living in the control (BzrDir) object must be locked for
        writing.

        :param control: BzrDir containing the branch reference
        :param location: value to write to the branch reference location.
        """
        branch_format = BranchFormat.find_format(control)
        transport = control.get_branch_transport(None)
        branch = branch_format.open(control)
        location = transport.put('location', location)
