"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GenericMLRouter = void 0;
var _zlib = require("zlib");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Generic ML client detector with caching for performance
 * Finds any client in the request context that has a request() method
 */
let cachedClientKey = null;
let detectionAttempted = false;
function findMLClient(context) {
  // Use cached result if available
  if (detectionAttempted) {
    if (cachedClientKey) {
      const contextItem = context[cachedClientKey];
      return contextItem === null || contextItem === void 0 ? void 0 : contextItem.client;
    }
    return undefined;
  }

  // First-time detection - cache the result
  for (const key of Object.keys(context)) {
    var _contextItem$client;
    const contextItem = context[key];
    if (contextItem !== null && contextItem !== void 0 && (_contextItem$client = contextItem.client) !== null && _contextItem$client !== void 0 && _contextItem$client.request && typeof contextItem.client.request === 'function') {
      cachedClientKey = key; // Cache the key name
      detectionAttempted = true;
      return contextItem.client;
    }
  }
  detectionAttempted = true; // Mark as attempted even if not found
  return undefined;
}

// ============================================================================
// GENERIC ML CLIENT TYPES - To avoid compile-time dependencies
// ============================================================================

/**
 * Type guard to check if response is streaming
 */
function isStreamResponse(response) {
  return response && typeof response.body === 'object' && 'pipe' in response.body;
}

/**
 * Generic ML Commons agent router
 * Uses generic ML client detection to communicate with ML Commons agents
 * Works with any ML client provider that has a request() method
 */
class GenericMLRouter {
  async forward(context, request, response, logger, configuredAgentId, dataSourceId, observabilityAgentId) {
    var _request$body$forward;
    // Validate request body
    if (!request.body || typeof request.body !== 'object') {
      return response.customError({
        statusCode: 400,
        body: {
          message: 'Invalid request body for ML Commons agent'
        }
      });
    }
    const language = (_request$body$forward = request.body.forwardedProps) === null || _request$body$forward === void 0 ? void 0 : _request$body$forward.queryAssistLanguage;
    const agentId = language ? observabilityAgentId : configuredAgentId;
    if (!agentId) {
      return response.customError({
        statusCode: 503,
        body: {
          message: 'ML Commons agent ID not configured'
        }
      });
    }
    let mlClient = findMLClient(context);
    if (!mlClient) {
      mlClient = {
        request: async params => {
          const client = dataSourceId && context.dataSource ? await context.dataSource.opensearch.getClient(dataSourceId) : context.core.opensearch.client.asCurrentUser;
          const result = await client.transport.request({
            method: params.method,
            path: params.path,
            body: params.body
          }, {
            asStream: params.stream,
            requestTimeout: params.timeout,
            maxRetries: 0
          });
          return {
            status: result.statusCode || 200,
            statusText: 'OK',
            headers: result.headers || {},
            body: result.body
          };
        }
      };
    }
    try {
      logger.info('Forwarding request to ML Commons agent', {
        agentId,
        language,
        dataSourceId
      });
      const mlResponse = await mlClient.request({
        method: 'POST',
        path: `/_plugins/_ml/agents/${agentId}/_execute/stream`,
        body: JSON.stringify(request.body),
        datasourceId: dataSourceId,
        // Use actual dataSourceId from request
        stream: true,
        timeout: 300000
      }, request, context);

      // Handle streaming response properly using type guard
      if (isStreamResponse(mlResponse)) {
        const contentEncoding = mlResponse.headers['content-encoding'] || mlResponse.headers['Content-Encoding'];
        const encHeader = Array.isArray(contentEncoding) ? contentEncoding.join(',') : contentEncoding !== null && contentEncoding !== void 0 ? contentEncoding : '';
        const encodings = encHeader.toLowerCase().split(',').map(e => e.trim());
        let responseBody = mlResponse.body;
        if (encodings.includes('gzip')) {
          const gunzip = (0, _zlib.createGunzip)();
          gunzip.on('error', err => {
            logger.error(`Gzip decompression error: ${err.message}`);
          });
          responseBody = mlResponse.body.pipe(gunzip);
        }
        return response.custom({
          statusCode: mlResponse.status,
          headers: {
            'Content-Type': 'text/event-stream',
            'Content-Encoding': 'identity',
            Connection: 'keep-alive'
          },
          body: responseBody
        });
      } else {
        return response.custom({
          statusCode: mlResponse.status,
          headers: {
            'Content-Type': 'application/json',
            ...mlResponse.headers
          },
          body: typeof mlResponse.body === 'string' ? JSON.parse(mlResponse.body) : mlResponse.body
        });
      }
    } catch (error) {
      logger.error(`Error forwarding to ML Commons agent: ${error}`);
      return response.customError({
        statusCode: (error === null || error === void 0 ? void 0 : error.status) || 500,
        body: {
          message: (error === null || error === void 0 ? void 0 : error.message) || 'Unknown error'
        }
      });
    }
  }
  getRouterName() {
    return 'GenericMLRouter';
  }
}
exports.GenericMLRouter = GenericMLRouter;