package Lire::Config::Scalar;

use strict;

use base qw/Lire::Config::Value/;

use Locale::TextDomain 'lire';

use Lire::Utils qw/ xml_encode shell_quote/;

use Carp;

=pod

=head1 NAME

Lire::Config::Scalar - Lire Config container for scalar values

=head1 SYNOPSIS

  use Lire::Config::Scalar

=head1 DESCRIPTION

This package contains the class to represent a scalar in an actual
configuration.

=head1 Lire::Config::Scalar

Value container for scalar values. Implements simple methods to get()
and set() string data.

=cut

sub new {
    my $self = shift->SUPER::new(@_);
    my %args = @_;

    $self->{'value'} = ''
      unless exists $self->{'value'};

    $self->{'warned'} = 0;

    $self->set( $args{'value'} )
      if exists $args{'value'};

    return $self;
}

sub as_value {
    my $self = $_[0];

    my $spec = $self->spec();

    my $val = $spec->normalize( $self->{'value'} );
    return $val if $spec->is_valid( $val );

    unless ( $self->{'warned'} ) {
        carp __x( "invalid value for parameter '{parameter}': {value}",
                  'parameter' => $self->name(),
                  'value' => ( defined $val ? $val : '(undef)' ) );
        $self->{'warned'} = 1;
    }

    return wantarray ? () : undef;
}

sub as_label {
    my $label = $_[0]->get();

    return '<undef>' unless defined $label;

    $label =~ s/(^\s*|\s*$)//g;

    return $label;
}

sub as_shell_var {
    my $self = $_[0];

    return ( $self->spec->is_valid( $self->{'value'} )
             ? $self->name . "=" . shell_quote( $self->as_value() )
             : '' );
}

=pod

=head2 get()

Returns this scalar's value.

=cut

sub get {
    return $_[0]{'value'};
}

=pod

=head2 set( $val )

This method can be used to change this scalar's value.

=cut

sub set {
    my ($self, $val) = @_;

    $self->{'warned'} = 0;

    $self->{'value'} = $val;
}

=pod

=head2 is_valid()

Checks with the parameter's spec that the current value is valid.

=cut

sub is_valid {
    return $_[0]->{'spec'}->is_valid( $_[0]->{'value'} );
}

sub is_equals {
    my ( $self, $param ) = @_;

    return 0 unless $self->SUPER::is_equals( $param );

    no warnings 'uninitialized';
    return $self->{'spec'}->normalize( $self->{'value'} ) eq
      $self->{'spec'}->normalize( $param->{'value'} );
}

sub save_xml {
    my ( $self, $fh, $indent, $xmlns ) = @_;

    $indent ||= 0;
    $xmlns ||= '';
    return if $self->is_default() || $self->spec()->obsolete();

    if(defined $self->{'value'}) {
        print $fh '  'x$indent, "<${xmlns}param name=\"", $self->name, "\">";
        print $fh xml_encode($self->{'value'}), "</${xmlns}param>\n";
    }
}

1; # whine, whine

__END__

=head1 AUTHOR

Wessel Dankers <wsl@logreport.org>

=head1 VERSION

$Id: Scalar.pm,v 1.16 2008/03/09 19:27:31 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2002-2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
