﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesis-video-archived-media/KinesisVideoArchivedMediaRequest.h>
#include <aws/kinesis-video-archived-media/KinesisVideoArchivedMedia_EXPORTS.h>
#include <aws/kinesis-video-archived-media/model/ClipFragmentSelector.h>

#include <utility>

namespace Aws {
namespace KinesisVideoArchivedMedia {
namespace Model {

/**
 */
class GetClipRequest : public KinesisVideoArchivedMediaRequest {
 public:
  AWS_KINESISVIDEOARCHIVEDMEDIA_API GetClipRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetClip"; }

  AWS_KINESISVIDEOARCHIVEDMEDIA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the stream for which to retrieve the media clip. </p> <p>You must
   * specify either the StreamName or the StreamARN. </p>
   */
  inline const Aws::String& GetStreamName() const { return m_streamName; }
  inline bool StreamNameHasBeenSet() const { return m_streamNameHasBeenSet; }
  template <typename StreamNameT = Aws::String>
  void SetStreamName(StreamNameT&& value) {
    m_streamNameHasBeenSet = true;
    m_streamName = std::forward<StreamNameT>(value);
  }
  template <typename StreamNameT = Aws::String>
  GetClipRequest& WithStreamName(StreamNameT&& value) {
    SetStreamName(std::forward<StreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the stream for which to retrieve the media
   * clip. </p> <p>You must specify either the StreamName or the StreamARN. </p>
   */
  inline const Aws::String& GetStreamARN() const { return m_streamARN; }
  inline bool StreamARNHasBeenSet() const { return m_streamARNHasBeenSet; }
  template <typename StreamARNT = Aws::String>
  void SetStreamARN(StreamARNT&& value) {
    m_streamARNHasBeenSet = true;
    m_streamARN = std::forward<StreamARNT>(value);
  }
  template <typename StreamARNT = Aws::String>
  GetClipRequest& WithStreamARN(StreamARNT&& value) {
    SetStreamARN(std::forward<StreamARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time range of the requested clip and the source of the timestamps.</p>
   */
  inline const ClipFragmentSelector& GetClipFragmentSelector() const { return m_clipFragmentSelector; }
  inline bool ClipFragmentSelectorHasBeenSet() const { return m_clipFragmentSelectorHasBeenSet; }
  template <typename ClipFragmentSelectorT = ClipFragmentSelector>
  void SetClipFragmentSelector(ClipFragmentSelectorT&& value) {
    m_clipFragmentSelectorHasBeenSet = true;
    m_clipFragmentSelector = std::forward<ClipFragmentSelectorT>(value);
  }
  template <typename ClipFragmentSelectorT = ClipFragmentSelector>
  GetClipRequest& WithClipFragmentSelector(ClipFragmentSelectorT&& value) {
    SetClipFragmentSelector(std::forward<ClipFragmentSelectorT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_streamName;

  Aws::String m_streamARN;

  ClipFragmentSelector m_clipFragmentSelector;
  bool m_streamNameHasBeenSet = false;
  bool m_streamARNHasBeenSet = false;
  bool m_clipFragmentSelectorHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisVideoArchivedMedia
}  // namespace Aws
