/*
 * This file is part of Transportgoals, which is a GameScript for OpenTTD
 * Copyright (C) 2011  Leif Linse
 *
 * Transportgoals is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * Transportgoals is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Transportgoals; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/* Import SuperLib for GameScript */
import("util.superlib", "SuperLib", 19);
Result <- SuperLib.Result;
Log <- SuperLib.Log;
Helper <- SuperLib.Helper;
ScoreList <- SuperLib.ScoreList;
Tile <- SuperLib.Tile;
Direction <- SuperLib.Direction;
Town <- SuperLib.Town;
Industry <- SuperLib.Industry;

GOAL_STEP_SIZE <- 5;

require("version.nut"); // get SELF_VERSION

//------------------------------------------------------------//
//                                                            //
//   Class for storage of company specific data               //
//                                                            //
//------------------------------------------------------------//

class CompanyData
{
	_company = null;
	_goal_id = null;
	_goal_value = null;
	_vehicle_count_average = 0; // average * 100
	_transported = 0;
	_hq_sign = null; // sign id of the sign sitting ontop of the HQ

	constructor(company)
	{
		this._company = company;
		this._goal_id = null;
		this._goal_value = null;
		this._vehicle_count_average = 0;
		this._transported = 0;
		this._hq_sign = null;
	}

	function CreateGoal();

	// For save/load
	function SaveToTable();
	static function CreateFromTable(table);
}

function CompanyData::CreateGoal()
{
	this._goal_id =
		GSGoal.New(this._company, GSText(GSText.STR_GOAL, this._goal_value), GSGoal.GT_NONE, 0);
}

function CompanyData::SaveToTable()
{
	local result = {
		company = this._company,
		goal_id = this._goal_id,
		goal_value = this._goal_value,
		vehicle_count_average = this._vehicle_count_average,
		transported = this._transported,
		hq_sign = this._hq_sign,
	};

	return result;
}

/* static */ function CompanyData::CreateFromTable(table)
{
	local result = CompanyData(table.company);

	result._goal_id = table.goal_id;
	result._goal_value = table.goal_value;
	result._vehicle_count_average = table.vehicle_count_average;
	result._transported = table.transported;
	result._hq_sign = table.hq_sign;

	return result;
}


//------------------------------------------------------------//
//                                                            //
//   Main class                                               //
//                                                            //
//------------------------------------------------------------//

class TestGame extends GSController 
{
	_per_company = null;

	_loaded_data = null; // assigned when loading a save game
	_loaded_from_version = null; // assigned when loading a save game
	constructor()
	{
		_per_company = [];
		
	}

	function Save();
	function Load(version, data);
}

function TestGame::Start()
{
	this.UpdateGoalList();

	Log.Info("Setup done", Log.LVL_INFO);

	// Wait for the game to start
	this.Sleep(1);

	local last_goal_setup = GSDate.GetCurrentDate();
	while (true) {
		local loop_start_tick = GSController.GetTick();

		// Handle events
		this.HandleEvents();

		// Check goals setup once a month to fix problems when events has been lost
		local current_date = GSDate.GetCurrentDate();
		if(current_date > last_goal_setup + 30)
		{
			this.UpdateGoalList();
			last_goal_setup = current_date;
		}

		// Has any goals been fulfilled?
		this.ScanGoals();


		// Loop with a frequency of one day
		local ticks_used = GSController.GetTick() - loop_start_tick;
		this.Sleep(Helper.Max(1, 5 * 74 - ticks_used));
	}
}

function TestGame::HandleEvents()
{
	if(GSEventController.IsEventWaiting())
	{
		local ev = GSEventController.GetNextEvent();

		if(ev == null)
			return;

		local ev_type = ev.GetEventType();
		if(ev_type == GSEvent.ET_COMPANY_NEW ||
				ev_type == GSEvent.ET_COMPANY_BANKRUPT ||
				ev_type == GSEvent.ET_COMPANY_MERGER)
		{
			Log.Info("A company was created/bankrupt/merged => update goal list", Log.LVL_INFO);

			// Update the goal list when:
			// - a new company has been created
			// - a company has gone bankrupt
			// - a company has been bought by another company
			this.UpdateGoalList();
		}
	}
}

// Add/remove goals to the goal list
function TestGame::UpdateGoalList()
{
	// Was a previous game loaded from a version that supports save/load?
	if(this._loaded_data != null && this._loaded_from_version > 1)
	{
		// Load company data from loaded data
		foreach(_, company_table in this._loaded_data)
		{
			if(company_table != null)
			{
				Log.Info("Loading data for company " + GSCompany.GetName(company_table.company), Log.LVL_INFO);
				this._per_company.append(
					CompanyData.CreateFromTable(company_table)
				);
			}
		}

		this._loaded_data = null; // don't attempt to load again
	}

	// Loop over all possible company IDs
	for(local c = GSCompany.COMPANY_FIRST; c <= GSCompany.COMPANY_LAST; c++)
	{
		// has goals already been set up for this company?
		local existing = null;
		local existing_idx = 0;
		foreach(company_data in this._per_company)
		{
			if(company_data._company == c)
			{
				existing = company_data;
				break;
			}
			existing_idx++;
		}

		// does the company exist in the game
		if(GSCompany.ResolveCompanyID(c) == GSCompany.COMPANY_INVALID)
		{
			if(existing != null)
			{
				// Remove data for no longer existing company
				this._per_company.remove(existing_idx);
			}
			continue;
		}
	
		// If the company can be resolved and exist (goals has been setup) => don't do anything
		if(existing != null) continue;

		// Company goals has not yet been setup for this company
		local company_data = CompanyData(c);
		company_data._goal_value = GOAL_STEP_SIZE;
		company_data.CreateGoal();

		this._per_company.append(company_data);
	}
}

// Has any goals been fulfilled?
function TestGame::ScanGoals()
{
	foreach(company_data in this._per_company)
	{
		local delivered = GSCompany.GetQuarterlyCargoDelivered(company_data._company, GSCompany.CURRENT_QUARTER + 1);
		local veh_count = GSVehicleList().Count();

		company_data._vehicle_count_average = (company_data._vehicle_count_average * 5 + veh_count * 100) / 6;

		local cargo_per_veh = company_data._vehicle_count_average == 0? 0 :
			delivered * 100 / company_data._vehicle_count_average;

		Log.Info("Cargo/vehicle for " + GSCompany.GetName(company_data._company) + ": " + cargo_per_veh, Log.LVL_DEBUG);

		this.UpdateHQSign(company_data);

		if(cargo_per_veh > company_data._goal_value)
		{
			GSGoal.Remove(company_data._goal_id);
			company_data._goal_value += GOAL_STEP_SIZE;
			company_data.CreateGoal();
			Log.Info(GSCompany.GetName(company_data._company) + " completed its goal", Log.LVL_INFO);
		}
	}
}

function TestGame::UpdateHQSign(company_data)
{
	// Put the sign on the south tile of the HQ
	local hq_tile = Direction.GetAdjacentTileInDirection(
			GSCompany.GetCompanyHQ(company_data._company),
			Direction.DIR_S);

	local hq_exist = GSMap.IsValidTile(hq_tile);
	if(company_data._hq_sign != null && !hq_exist)
	{
		// Sign exist, but no HQ => remove sign
		GSSign.RemoveSign(company_data._hq_sign);
		company_data._hq_sign = null;
	}
	else if(hq_exist)
	{
		// The HQ exist
		local sign_text = GSText(GSText.STR_SCORE, (company_data._goal_value - GOAL_STEP_SIZE));
		if(company_data._hq_sign == null)
		{
			// HQ exist, but no sign yet => create new sign
			company_data._hq_sign = GSSign.BuildSign(hq_tile, sign_text);
		}
		else
		{
			// HQ exist as well as a sign
			if(GSSign.GetLocation(company_data._hq_sign) == hq_tile)
			{
				// The sign is at the right location => update only the text contents
				GSSign.SetName(company_data._hq_sign, sign_text);
			}
			else
			{
				// The sign exist, but at the wrong tile
				GSSign.RemoveSign(company_data._hq_sign);
				company_data._hq_sign = GSSign.BuildSign(hq_tile, sign_text);
			}
		}
	}
	else
	{
		// No HQ and no old sign
	}

}

function TestGame::Save()
{
	Log.Info("Saving data", Log.LVL_SUB_DECISIONS);
	local result = {};

	foreach(company_data in this._per_company)
	{
		result[company_data._company] <- company_data.SaveToTable();
	}

	foreach(i, k in result)
	{
		Log.Info(i + " has " + k);
	}

	return result;
}

function TestGame::Load(version, table)
{
	Log.Info("Loading..", Log.LVL_INFO);
	Log.Info("Previously saved with Game Script version " + version, Log.LVL_INFO);

	if(version > SELF_VERSION)
	{
		Log.Warning("Warning: Loading from a newer version of TransportGoals", Log.LVL_INFO);
	}
	if(version == 1)
	{
		Log.Warning("Loading from version 1 of TransportGoals.", Log.LVL_INFO);
	   	Log.Warning("Version 1 did not support save/load, so you will get duplicate goals.", Log.LVL_INFO);
	}
	this._loaded_data = table;
	this._loaded_from_version = version;
}
