/* -*- Mode: C++; tab-width: 4 -*- */
/**
 *    This file is part of Awards
 *    (c) krinn@chez.com
 *
 *    It's free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    any later version.
 *
 *    You should have received a copy of the GNU General Public License
 *    with it.  If not, see <http://www.gnu.org/licenses/>.
 *
**/

class Utils
{
	constructor()
	{
	}
}

/** @brief Convert an array or GSList to a string
 *
 * @param list an array or an GSList
 * @return a string with item and value in it
 *
 */
function Utils::ArrayListToString(list)
{
	local ret = "[";
	local check = false;
	if (typeof(list) == "array")	{ check = (list.len() == 0); }
							else	{ check = list.IsEmpty(); }
	if (!check)
			{
			local contents = "";
			foreach (a, b in list)
				{
				if (contents != "") contents += ", ";
				if (b != -1)	contents += a + " = " + b;
				}
			ret += contents;
			}
	ret += "]";
	return ret;
}

/** @brief Convert a GSList to an array
 *
 * @param list a GSList
 * @return Array
 *
 */
function Utils::GSListToArray(list)
{
	local ar = [];
	foreach (item, value in list)	{ ar.push(item); ar.push(value); }
	return ar;
}

/** @brief Convert an array to a GSList
 *
 * @param ar an Array
 * @return GSList
 *
 */
function Utils::ArrayToGSList(ar)
{
	local list = GSList();
	for (local i = 0; i < ar.len(); i++)
		{
		list.AddItem(i, i+1);
		i++;
		}
	return list;
}

/** @brief Check if an item is in an array
 *
 * @param seek The item to check presence
 * @param data The array itself
 * @return -1 if item is not in array, else position of item in array
 *
 */
function Utils::INArray(seek, data)
{
	foreach (item, value in data)	{ if (value == seek)	{ return item; } }
	return -1;
}


class Cache
{
static cargo = GSList();   // item cargoID of passenger, value unused
static vehicle = array(17, -1); // 1-14 list of vehicle per company, 15 - vehicle list of all companies, 16 - date: to dirty cache
static snowtown = GSList(); // item = townID of town with snow, value unused
static powerplant = GSList(); // item : industryID that handle coal/energy cargo, value = cargoID for coal: item=0, value = -1 if none have any
static cargo_handle = GSList();	// item company, value = bit set of cargo handle by company
static cargo_town = GSList(); // item: cargoID with town effect, value unused
static cargo_industry = GSList(); // item : cargoID that are produce by industries, value : unused
static cargo_tracker = ["WOOD", "OIL_", "TOYS", "WATR", "GOOD", "DIAM"];
static bigarray = [];
static indexer = {};
	id	= null;
	index = null;

	constructor(strid, defvalue = 0)
	{
		this.id = strid;
		if (this.id in Cache.indexer)	{ GSLog.Info(strid+" is already index"); return; }
		this.index = Cache.bigarray.len();
		for (local i = 0; i < 15; i++)	{ Cache.bigarray.push(defvalue); }
		GSLog.Info("New entry "+this.id+" at index "+this.index);
		Cache.indexer[this.id] <- this;
	}
}

/** @brief Set the cache data
 *
 * @param strID The cache string identifier
 * @param companyID the companyID
 * @param data The value to set
 *
 */
function Cache::SetData(strID, companyID, data)
{
	if (!(strID in Cache.indexer))	{ return -1; }
	local goal = Cache.indexer[strID];
	Cache.bigarray[goal.index + companyID] = data;
}

/** @brief Get data from cache
 *
 * @param strID The cache string identifier
 * @param companyID The companyID
 * @return The seeked data
 *
 */
function Cache::GetData(strID, companyID)
{
	if (!(strID in Cache.indexer))	{ return -1; }
	local goal = Cache.indexer[strID];
	return (Cache.bigarray[goal.index + companyID]);
}

/** @brief Get the first element that index that identifier in the cache
 *
 * @param strID The cache string identifier
 * @return index of company0
 *
 */
function Cache::GetIndex(strID)
{
	if (!(strID in Cache.indexer))	{ return -1; }
	local goal = Cache.indexer[strID];
	return goal.index;
}

/** @brief Calculate the company value seeking its balance and value of its vehicles
 *
 * @param companyID The companyID
 * @return the company value
 *
 */
function Cache::CalcCompanyValue(companyID)
{
	local cmode = GSCompanyMode(companyID);
	local value = GSCompany.GetBankBalance(companyID);
	value -= GSCompany.GetLoanAmount();
	local vlist = Cache.Vehicle_Helper();
	vlist[companyID].Valuate(GSVehicle.GetCurrentValue);
	foreach (veh, price in vlist[companyID])	{ value += price; }
	return value;
}

/** @brief Get a list of valid companies in the game
 *
 * @return An array with only valid companies in it
 *
 */
function Cache::Company_Helper()
{
	local company = [];
	for (local j = 0; j < 15; j++)
		{
		local c_idx = GSCompany.ResolveCompanyID(j);
		if (c_idx != GSCompany.COMPANY_INVALID)	{ company.push(c_idx); }
		}
	return company;
}

/** @brief Get the vehicle list own by companies
 *
 * @return Array idx 0-14 GSList vehicle per company, [15] = GSList of all vehicle. Make sure you don't alter that list or force dirty cache to rebuild it!!!
 *
 */
function Cache::Vehicle_Helper()
{
	local d = Cache.vehicle[16];
	local now = GSDate.GetCurrentDate();
	if ((now - d) < 10)	{ return Cache.vehicle; }
	local vlist = GSVehicleList();
	vlist.Valuate(GSVehicle.GetOwner);
	Cache.vehicle[15].Clear();
	Cache.vehicle[15].AddList(vlist);
	for (local i = 0; i < 15; i++)
		{
		Cache.vehicle[i].Clear();
		Cache.vehicle[i].AddList(vlist);
		Cache.vehicle[i].KeepValue(i);
		}
	Cache.vehicle[16] = GSDate.GetCurrentDate();
	return Cache.vehicle;
}

/** @brief Get the number of vehicle type a company have
 *
 * @param v_str Vehicle to count : "train" "boat" "bus" "truck" "aircraft" "chopper" and "tram"
 * @return GSList with all vehicle that match critera, index: the companyID, value: the count
 *
 */
function Cache::Vehicle_Helper_Counter(v_str)
{
	local vlist = Cache.Vehicle_Helper();
	local company = Cache.Company_Helper();
	local retvalue = GSList();
	for (local j = 0; j < 15; j++)	{ retvalue.AddItem(j, 0); }
	local v_type = null;
	foreach (comp in company)
		{
		local tl = GSList();
		tl.AddList(vlist[comp]);
		tl.Valuate(GSVehicle.GetVehicleType);
		if (v_str == "train" || v_str == "boat")
				{
				if (v_str == "train")	{ tl.KeepValue(GSVehicle.VT_RAIL); }
								else	{ tl.KeepValue(GSVehicle.VT_WATER); }
				}
		else	{ // air & road
				if (v_str == "bus" || v_str == "truck"  || v_str == "tram")
						{
						tl.KeepValue(GSVehicle.VT_ROAD);
						local pass = Cache.GetPassengerCargo();
						local filter = GSList();
						foreach (veh, _ in tl)
							{
							filter.AddItem(veh, 0); // add as truck
							foreach (pcargo, _ in pass)
								{
								if (GSVehicle.GetCapacity(veh, pcargo) > 0)	{ filter.SetValue(veh, 1); }
								if (GSVehicle.GetRoadType(veh) == GSRoad.ROADTYPE_TRAM)	{ filter.SetValue(veh, 2); }
								}
							}
						if (v_str == "bus")	{ filter.KeepValue(1); }
									else	if (v_str == "truck") { filter.KeepValue(0); }
												else { filter.KeepValue(2); }
						tl.Clear();
						tl.AddList(filter);
						}
				else	{
						tl.KeepValue(GSVehicle.VT_AIR);
						local filter = GSList();
						foreach (veh, _ in tl)
							{
							filter.AddItem(veh, 0); // plane are 0
							local eng = GSVehicle.GetEngineType(veh);
							if (GSEngine.GetPlaneType(eng) == GSAirport.PT_HELICOPTER)	{ filter.SetValue(veh, 1); }
							}
						if (v_str == "aircraft")	{ filter.KeepValue(0); }
											else	{ filter.KeepValue(1); }
						tl.Clear();
						tl.AddList(filter);
						}
				}
		retvalue.SetValue(comp, tl.Count());
		}
	return retvalue;
}

/** @brief Return a GSList of cargo that are passengers
 *
 * @return GSList
 *
 */
function Cache::GetPassengerCargo()
	{
	if (!Cache.cargo.IsEmpty())	{ return Cache.cargo; }
	local cargolist = GSCargoList();
	foreach (cargo, dummy in cargolist)
			{
			if (GSCargo.GetTownEffect(cargo) == GSCargo.TE_PASSENGERS) { Cache.cargo.AddItem(cargo, 0); }
			}
	return Cache.cargo;
	}

/** @brief Return a GSList of town that are under snow
 *
 * @param false addnew = true to rebuild the list, false to grab the current cache one
 * @return GSList
 *
 */
function Cache::GetSnowTown(addnew = false)
{
	if (!Cache.snowtown.IsEmpty() && !addnew)	{ return Cache.snowtown; }
	local town_list = GSTownList();
	town_list.RemoveList(Cache.snowtown);
	local terrain = GSTileList();
	foreach (town, _ in town_list)
		{
		terrain.Clear();
		local location = GSTown.GetLocation(town);
	    terrain.AddRectangle(location + GSMap.GetTileIndex(-8, -8), location + GSMap.GetTileIndex(8, 8));
		terrain.Valuate(GSTile.GetTerrainType);
		terrain.KeepValue(GSTile.TERRAIN_SNOW);
		if (!terrain.IsEmpty())	{
								Cache.snowtown.AddItem(town, 0);
								}
		}
	return Cache.snowtown;
}

/** @brief Return a GSList of industries that produce energy
 *
 * @param false addnew = true to rebuild the list, false to grab the current cache one
 * @return GSList
 *
 */
function Cache::GetPowerPlant(addnew = false)
{
	if (!Cache.powerplant.IsEmpty() && !addnew && Cache.powerplant.GetValue(0) != -1)	{ return Cache.powerplant; }
	local cargolabel = ["COAL"];
	local cargo_list = GSCargoList();
	local energy_list = GSList();
	if (Cache.powerplant.HasItem(0))	{ Cache.powerplant.RemoveItem(0); }
	foreach (cargoid, _ in cargo_list)
		{
		local t_list = GSList();
		if (Utils.INArray(GSCargo.GetCargoLabel(cargoid), cargolabel) != -1)
			{
			t_list = GSIndustryList_CargoAccepting(cargoid);
			foreach (indID, _ in t_list)
				{
				Cache.powerplant.AddItem(indID, cargoid);
				}
			}
		}
	if (Cache.powerplant.IsEmpty())	{ Cache.powerplant.AddItem(0, -1); }
	return Cache.powerplant;
}

/** @brief Like AIIndustry::GetAmountOfStationAround but doesn't count our stations, so we only grab competitors stations
 * return 0 or numbers of stations not own by owner near the place, adapt from DictatorAI
 *
 * @param IndustryID The industryID to check
 * @return GSList
 *
 */
function Cache::GetCompetitorStationAround(IndustryID)
{
	local comp_list = GSList();
	local counter=0;
	local place = GSIndustry.GetLocation(IndustryID);
	local radius = GSStation.GetCoverageRadius(GSStation.STATION_TRAIN);
	local tiles = GSTileList();
	local produce = GSTileList_IndustryAccepting(IndustryID, radius);
	local accept = GSTileList_IndustryProducing(IndustryID, radius);
	tiles.AddList(produce);
	tiles.AddList(accept);
	tiles.Valuate(GSTile.IsStationTile); // force keeping only station tile
	tiles.KeepValue(1);
	tiles.Valuate(GSStation.GetStationID);
	foreach (tile, stationID in tiles)
		{ // remove duplicate id
		if (!comp_list.HasItem(stationID))	{ comp_list.AddItem(stationID, GSStation.GetOwner(stationID)); }
		}
	return comp_list;
}

/** @brief Return GSList of companies infrastructure infos
 *
 * @param inftype The type of infrastructure to check : one of GSInfrastructure.Infrastructure type
 * @return GSList
 *
 */
function Cache::Infrastructure_helper(inftype)
{
	local comp_list = Cache.Company_Helper();
	local ret = GSList();
	foreach (comp in comp_list)
		{
		local k = GSInfrastructure.GetInfrastructurePieceCount(comp, inftype);
		ret.AddItem(comp, k);
		}
	return ret;
}

/** @brief Fill the cache value of cargo_town and cargo_industry with cargo that are for town or for industry
 *
 *
 */
function Cache::FindCargoUsage()
{
	if (!Cache.cargo_town.IsEmpty())	{ return; }
	local indtype_list = GSIndustryTypeList();
	foreach (item, value in indtype_list)
		{
        local cargolist = GSIndustryType.GetAcceptedCargo(item);
		foreach (cargo, _ in cargolist)
			{
			if (!Cache.cargo_industry.HasItem(cargo))	{ Cache.cargo_industry.AddItem(cargo, item); }
			}
		}
	local cargo_list = GSCargoList();
	Cache.cargo_town.AddList(cargo_list);
	foreach (cargo, _ in cargo_list)
		{
		if (GSCargo.GetTownEffect(cargo) == GSCargo.TE_NONE)	{ Cache.cargo_town.RemoveItem(cargo); }
		local label = GSCargo.GetCargoLabel(cargo);
		if (Utils.INArray(label, Cache.cargo_tracker) != -1)	{ local entry = Cache(label, 0); }
		}
	local o_str = [];
	foreach (cargo, _ in Cache.cargo_town)	{ o_str.push(GSCargo.GetCargoLabel(cargo)); }
	GSLog.Info("Towns accept cargo : "+Utils.ArrayListToString(o_str));
	o_str = [];
	foreach (cargo, _ in Cache.cargo_industry)	{ o_str.push(GSCargo.GetCargoLabel(cargo)); }
	GSLog.Info("Industries accept cargo : "+Utils.ArrayListToString(o_str));
}

/** @brief Monitor cargo handle by companies
 *
 *
 */
function Cache::Monitoring()
{
	local companies = Cache.Company_Helper();
	local town_list = GSTownList();
	foreach (company in companies) // Insane loops !
		{
		local stown = Cache.GetData("supply_town", company);
		if (stown < 1000000)
			{
			foreach (town, _ in town_list)
				{
				if (GSTown.GetRating(town, company) != 0)
					{
					foreach (cargo, _ in Cache.cargo_town)
						{
						local z = GSCargoMonitor.GetTownDeliveryAmount(company, cargo, town, true);
						if (z > 0)	{
									local k = Cache.cargo_handle.GetValue(company);
									k = (k | (1 << cargo));
									Cache.cargo_handle.SetValue(company, k);
									}
						if (GSCargo.GetCargoLabel(cargo) == "GOOD") // special monitoring of good
							{
							local h = Cache.GetData("GOOD", company);
							h += z;
							Cache.SetData("GOOD", company, h);
							}
						stown += z;
						if (stown > 1000000)	{ GSCargoMonitor.GetTownDeliveryAmount(company, cargo, town, false); }
						Cache.SetData("supply_town", company, stown);
						}
					}
				}
			}
		local sind = Cache.GetData("supply_industry", company);
		if (sind < 1000000)
			{
			foreach (cargo, _ in Cache.cargo_industry)
				{
				local ind_list = GSIndustryList_CargoAccepting(cargo);
				foreach (industry, iii in ind_list)
					{
					local z = GSCargoMonitor.GetIndustryDeliveryAmount(company, cargo, industry, true);
					if (z > 0)	{
								local k = Cache.cargo_handle.GetValue(company);
								k = (k | (1 << cargo));
								Cache.cargo_handle.SetValue(company, k);
								}
					local hh = GSCargo.GetCargoLabel(cargo);
					if (Utils.INArray(hh, Cache.cargo_tracker) != -1)  // special monitoring of specific cargos
						{
						local h = Cache.GetData(hh, company);
						h += z;
						Cache.SetData(hh, company, h);
						}
					sind += z;
					if (sind > 1000000)	{ GSCargoMonitor.GetIndustryDeliveryAmount(company, cargo, industry, false); }
					Cache.SetData("supply_industry", company, sind);
					}
				}
			}
		}
}

/** @brief Add a company that get an award for delivering cargos and grant it
 *
 * @param number The AwardID
 * @param comp The companyID
  *
 */
function Cache::DeliveryAwardGeneric(number, comp)
{
	if (Awards.HaveAward(number, comp))	{ return; }
	local awd = Awards.Get(number);
	if (awd == false)	{ return; }
	local everyone = [];
	everyone.extend(awd.Own_By_Company);
	everyone.push(comp);
	Awards.GrantAward(number, everyone);
}

