/* 
  TinyUZ2 - UZ2 (de)Compression Utility
  version 1.2.1, February 28, 2005

  Copyright (C) 2004, 2005 Michiel Hendriks

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  Michiel Hendriks - elmuerte@drunksnipers.com
*/

#include "tinyuz2.h"

#define BLOCKSIZE 32768
#define COMPSIZE  33096
#define USIGN 0x9E2A83C1

int verbose = 1;
int testmode = 0;
char *iname, *oname, *outpath;
FILE *ifile, *ofile;

void printHeader ()
{
  printf
    ("TinyUZ2 v1.2.1 - copyright (c) 2004, 2005 Michiel Hendriks <elmuerte@drunksnipers.com>\n");
  printf ("\n");
}

void printHelp (char *cmd)
{
  printf ("Usage:\n");
  printf ("\t%s [-c|-d|-s|-h] [-v|-q] [-t] [-o <path>] file ...\n\n", cmd);
  printf (" -c\t\tcompress file (default)\n");
  printf (" -d\t\tdecompress file\n");
  printf (" -h\t\tthis message\n");
  printf (" -o <path>\tset output path\n");
  printf (" -s\t\tshow file information (GUID, GEN, ...)\n");
  printf (" -t\t\ttest mode, don't create output files\n");
  printf (" -q\t\tquiet\n");
  printf (" -v\t\tincrease verbosity\n");
}

/** 
    return just the filename
*/
char *basename (char *path)
{
  char *bname;
  bname = (char *) strrchr (path, PATHDELIM);
  if (!bname)
    bname = path;
  else
    bname++;
  return bname;
}

/** 
    compress a file, returns non zero is failed
*/
int compressFile ()
{
  uLong res;
  uLong usize, csize;
  uLong isize, osize;
  long sign;
  Byte cbuf[COMPSIZE];          // compressed
  Byte ubuf[BLOCKSIZE];         // uncompressed

  // check if it's a package
  fread (&sign, 1, sizeof (sign), ifile);
  if (BOS_LONG (sign) != USIGN)
  {
    if (verbose > 0)
      fprintf (stderr, "%s is not an UnrealEngine package\n", iname);
    return -1;
  }

  // get file size
  fseek (ifile, 0, SEEK_END);
  isize = ftell (ifile);
  rewind (ifile);

  if (verbose > 0)
  {
    printf ("Compressing file: %s (%i bytes) -> %s ", iname, isize, oname);
    fflush (stdout);
  }

  while ((usize = fread (ubuf, 1, BLOCKSIZE, ifile)) > 0)
  {
    csize = BLOCKSIZE;
    fflush (stdout);
    if ((res = compress (cbuf, &csize, ubuf, usize)) != Z_OK)
    {
      if (verbose > 0)
        fprintf (stderr, "\nCompression error: %i\n", res);
      return res;
    }
    BOS_LONG_S (csize);
    fwrite (&csize, sizeof (csize), 1, ofile);
    BOS_LONG_S (usize);
    fwrite (&usize, sizeof (usize), 1, ofile);
    fwrite (cbuf, 1, BOS_LONG (csize), ofile);
  }
  osize = ftell (ofile);
  if (verbose > 0)
    printf ("(%i bytes) = %i%%\n", osize, (osize * 100 / isize));

  return 0;
}

/** 
    uncompress a file, returns non zero is failed
*/
int uncompressFile ()
{
  uLong res;
  uLong usize, csize;
  uLong isize, osize;
  long sign;
  Byte cbuf[COMPSIZE];          // compressed
  Byte ubuf[BLOCKSIZE];         // uncompressed

  // get file size
  fseek (ifile, 0, SEEK_END);
  isize = ftell (ifile);
  rewind (ifile);

  if (isize <= 0)
  {
    if (verbose > 0)
      fprintf (stderr, "%s : filesize is <= 0\n", iname);
    return -1;
  }
  if (verbose > 0)
  {
    printf ("Uncompressing file: %s (%i bytes) -> %s ", iname, isize, oname);
    fflush (stdout);
  }

  while (feof (ifile) == 0)
  {
    fread (&csize, sizeof (csize), 1, ifile);
    BOS_LONG_S (csize);
    if (feof (ifile))
      break;
    fread (&usize, sizeof (usize), 1, ifile);
    BOS_LONG_S (usize);
    if ((csize > COMPSIZE) || (usize > BLOCKSIZE))
    {
      if (verbose > 0)
        fprintf (stderr, "\nNot a valid uz2 file\n");
      return -1;
    }
    fflush (stdout);
    res = fread (cbuf, 1, csize, ifile);
    if (res != csize)
    {
      if (verbose > 0)
        fprintf (stderr, "\nRead error from stream: %i/%i\n", res, csize);
      return -1;
    }
    if ((res = uncompress (ubuf, &usize, cbuf, csize)) != Z_OK)
    {
      if (verbose > 0)
        fprintf (stderr, "\nUncompress error %i\n", res);
      return res;
    }
    fwrite (ubuf, 1, usize, ofile);
  }

  osize = ftell (ofile);
  if (verbose > 0)
    printf ("(%i bytes) = %i%%\n", osize, (osize * 100 / isize));

  // check if it's a package
  rewind (ofile);
  fread (&sign, 1, sizeof (sign), ofile);
  if (BOS_LONG (sign) != USIGN)
  {
    if (verbose > 0)
      fprintf (stderr, "%s is not an UnrealPackage\n", oname);
    remove (oname);
    return -1;
  }

  return 0;
}

/**
   Decompress the first block and pass it to showInfo
 */
int showDecompressInfo (char *uname, FILE * ufile)
{
  uLong res;
  uLong usize, csize;
  Byte cbuf[COMPSIZE];          // compressed
  Byte ubuf[BLOCKSIZE];         // uncompressed
  FILE *uufile;

  uufile = tmpfile ();
  rewind (ufile);

  fread (&csize, sizeof (csize), 1, ufile);
  BOS_LONG_S (csize);
  fread (&usize, sizeof (usize), 1, ufile);
  BOS_LONG_S (usize);
  if ((csize > COMPSIZE) || (usize > BLOCKSIZE))
    return -1;

  res = fread (cbuf, 1, csize, ufile);
  if (res != csize)
    return -1;
  if ((res = uncompress (ubuf, &usize, cbuf, csize)) != Z_OK)
    return res;
  fwrite (ubuf, 1, usize, uufile);

  res = showInfo (uname, uufile);
  fclose (uufile);
  return (int) res;
}

/**
   Show file information, supports compressed files
 */
int showInfo (char *uname, FILE * ufile)
{
  long sign, gen;
  long guid[4];
  int res;

  rewind (ufile);
  fread (&sign, 1, sizeof (sign), ufile);
  if (BOS_LONG (sign) != USIGN)
  {
    res = -1;
    // try to decompress
    fread (&gen, 1, sizeof (gen), ufile);
    if ((sign <= COMPSIZE) || (gen == BLOCKSIZE))
    {
      res = showDecompressInfo (uname, ufile);
    }
    if ((res != 0) && (verbose > 0))
      fprintf (stderr, "%s is not an UnrealPackage (%08X vs %08X)\n", uname,
               BOS_LONG (sign), USIGN);
    return res;
  }
  fread (&sign, 1, sizeof (sign), ufile);
  fseek (ufile, 28, SEEK_CUR);
  fread (&guid, 4, sizeof (long), ufile);
  fread (&gen, 1, sizeof (gen), ufile);
  printf ("%s : PVER=%i LVER=%i GUID=%08X%08X%08X%08X GEN=%i\n", uname,
          (BOS_LONG (sign) & 0xFFFF), (BOS_LONG (sign) >> 16),
          BOS_LONG (guid[0]), BOS_LONG (guid[1]), BOS_LONG (guid[2]),
          BOS_LONG (guid[3]), BOS_LONG (gen));
  return 0;
}

/**
   Main ...
 */
int main (int argc, char *argv[])
{
  int action = 0;
  int i, lastRes;
  int res = 0;
  char *tname;

  oname = (char *) malloc (1024);
  tname = (char *) malloc (1024);

  for (i = 1; i < argc; i++)
  {
    if (argv[i][0] == '-')
    {
      if (strcmp (argv[i], "-c") == 0)
        action = 0;
      else if (strcmp (argv[i], "-d") == 0)
        action = 1;
      else if (strcmp (argv[i], "-s") == 0)
        action = 2;
      else if (strcmp (argv[i], "-q") == 0)
        verbose = 0;
      else if (strcmp (argv[i], "-v") == 0)
        verbose = 2;
      else if (strcmp (argv[i], "-o") == 0)
        outpath = argv[++i];
      else if (strcmp (argv[i], "-t") == 0)
        testmode = 1;
      else if (strcmp (argv[i], "-h") == 0)
        action = -1;
      else
      {
        if (verbose > 0)
        {
          fprintf (stderr, "Invalid switch: %s\n", argv[i]);
          fprintf (stderr, "Use `%s -h` for more information\n", argv[0]);
        }
        return 1;
      }
    }
    else
      break;
  }
  if (i == argc)
    action = -1;
  if (verbose >= 1)
    printHeader ();

  if (action == -1)
  {
    if (verbose >= 1)
      printHelp (argv[0]);
    return -1;
  }
  else
  {
    for (; i < argc; i++)
    {
      switch (action)
      {
      case 0:                  // compress
        iname = argv[i];
        sprintf (oname, "%s.uz2", iname);
        if (outpath)
        {
          sprintf (tname, "%s%c%s", outpath, PATHDELIM, basename (oname));
          strcpy (oname, tname);
        }
        ifile = fopen (iname, "rb");
        if (!ifile)
        {
          res--;
          if (verbose > 0)
            fprintf (stderr, "Error opening input file %s\n", iname);
          break;
        }
        if (testmode)
          ofile = tmpfile ();
        else
          ofile = fopen (oname, "wb");
        if (!ofile)
        {
          res--;
          if (verbose > 0)
            fprintf (stderr, "Error opening output file %s\n", oname);
          break;
        }
        lastRes = compressFile ();
        res += lastRes;
        fclose (ifile);
        fclose (ofile);
        if (lastRes)
          unlink (oname);
        break;

      case 1:                  // decompress
        iname = argv[i];
        tname = (char *) strstr (iname, ".uz2");
        if (tname)
        {
          strncpy (oname, iname, strlen (iname) - strlen (tname));
          oname[strlen (iname) - strlen (tname)] = '\0';
        }
        else
          sprintf (oname, "%s.uxx", iname);
        if (outpath)
        {
          sprintf (oname, "%s%c%s", outpath, PATHDELIM, basename (oname));
        }
        ifile = fopen (iname, "rb");
        if (!ifile)
        {
          res--;
          if (verbose > 0)
            fprintf (stderr, "Error opening input file %s\n", iname);
          break;
        }
        if (testmode)
          ofile = tmpfile ();
        else
          ofile = fopen (oname, "wb+");
        if (!ofile)
        {
          res--;
          if (verbose > 0)
            fprintf (stderr, "Error opening output file %s\n", oname);
          break;
        }
        lastRes = uncompressFile ();
        res += lastRes;
        fclose (ifile);
        fclose (ofile);
        if (lastRes)
          unlink (oname);
        break;

      case 2:                  // showinfo
        iname = argv[i];
        ifile = fopen (iname, "rb");
        if (!ifile)
        {
          res--;
          if (verbose > 0)
            fprintf (stderr, "Error opening input file %s\n", iname);
          break;
        }
        res += showInfo (iname, ifile);
        fclose (ifile);

      }
    }

  }
  return res;
}
